<?php
declare(strict_types=1);

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/utils.php';

function hash_token(string $token): string {
  return hash('sha256', $token);
}
function require_owner(): array {
  $u = require_role('owner');
  require_password_changed_for_owner($u);
  return $u;
}


function generate_token_string(int $bytes = 32): string {
  return bin2hex(random_bytes($bytes));
}

function create_auth_token(int $userId, int $ttlSeconds = 604800): array {
  $token = generate_token_string(32);
  $tokenHash = hash_token($token);
  $expiresAt = gmdate('Y-m-d H:i:s', time() + $ttlSeconds);

  $ua = $_SERVER['HTTP_USER_AGENT'] ?? null;
  if (is_string($ua) && strlen($ua) > 255) $ua = substr($ua, 0, 255);

  $ip = $_SERVER['REMOTE_ADDR'] ?? null;
  if (is_string($ip) && strlen($ip) > 45) $ip = substr($ip, 0, 45);

  $stmt = db()->prepare("
    INSERT INTO auth_tokens (user_id, token_hash, expires_at, created_at, user_agent, ip_address)
    VALUES (?, ?, ?, ?, ?, ?)
  ");
  $stmt->execute([$userId, $tokenHash, $expiresAt, now_utc(), $ua, $ip]);

  return ['token' => $token, 'expires_at' => $expiresAt];
}

function get_bearer_token(): ?string {
  $auth = null;

  if (!empty($_SERVER['HTTP_AUTHORIZATION'])) {
    $auth = $_SERVER['HTTP_AUTHORIZATION'];
  } elseif (!empty($_SERVER['AUTHORIZATION'])) {
    $auth = $_SERVER['AUTHORIZATION'];
  } elseif (!empty($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
    $auth = $_SERVER['REDIRECT_HTTP_AUTHORIZATION'];
  } elseif (!empty($_SERVER['HTTP_X_AUTHORIZATION'])) {
    $auth = $_SERVER['HTTP_X_AUTHORIZATION'];
  } elseif (!empty($_SERVER['HTTP_X_AUTH_TOKEN'])) {
    $auth = $_SERVER['HTTP_X_AUTH_TOKEN'];
  }

  if (!$auth && function_exists('getallheaders')) {
    $headers = getallheaders();
    if (is_array($headers)) {
      foreach ($headers as $k => $v) {
        if (strtolower((string)$k) === 'authorization') {
          $auth = $v;
          break;
        }
      }
    }
  }

  if (!is_string($auth) || trim($auth) === '') $auth = null;

  if (is_string($auth)) {
    if (preg_match('/Bearer\s+(.+)/i', $auth, $m)) {
      $t = trim($m[1]);
      return $t !== '' ? $t : null;
    }
    $t = trim($auth);
    if ($t !== '') return $t;
  }

  $qToken = $_GET['access_token'] ?? $_GET['token'] ?? null;
  if (is_string($qToken)) {
    $t = trim($qToken);
    return $t !== '' ? $t : null;
  }

  return null;
}

function current_user(): ?array {
  static $cached = null;
  if ($cached !== null) return $cached;

  $token = get_bearer_token();
  if (!$token) return $cached = null;

  $tokenHash = hash_token($token);

 $stmt = db()->prepare("
  SELECT
    u.id,
    u.email,
    u.role,
    u.full_name,
    u.phone,
    u.must_change_password,
    u.is_active,
    t.expires_at,
    t.revoked_at
  FROM auth_tokens t
  JOIN users u ON u.id = t.user_id
  WHERE t.token_hash = ?
    AND t.revoked_at IS NULL
    AND t.expires_at > ?
  LIMIT 1
");

$stmt->execute([$tokenHash, now_utc()]);
$row = $stmt->fetch();

if (!$row) return $cached = null;
if ((int)($row['is_active'] ?? 0) !== 1) return $cached = null;

return $cached = $row;

}

function require_login(): array {
  $u = current_user();
  if (!$u) json_response(['ok' => false, 'error' => 'UNAUTHORIZED'], 401);
  return $u;
}

function require_role(string $role): array {
  $u = require_login();
  if (($u['role'] ?? '') !== $role) json_response(['ok' => false, 'error' => 'FORBIDDEN'], 403);
  return $u;
}

function require_password_changed_for_owner(array $user): void {
  if (($user['role'] ?? '') === 'owner' && (int)($user['must_change_password'] ?? 0) === 1) {
    json_response(['ok' => false, 'error' => 'MUST_CHANGE_PASSWORD'], 403);
  }
}

function revoke_token(string $token): void {
  $tokenHash = hash_token($token);
  $stmt = db()->prepare("UPDATE auth_tokens SET revoked_at = ? WHERE token_hash = ? AND revoked_at IS NULL");
  $stmt->execute([now_utc(), $tokenHash]);
}
