<?php
declare(strict_types=1);

require_once __DIR__ . '/../utils.php';
require_once __DIR__ . '/../db.php';

$body = get_request_json();

$shopId = (int)($body['shop_id'] ?? 0);
$customerPhone = safe_string($body['customer_phone'] ?? '', 30);
$referralPhone = safe_string($body['referral_phone'] ?? '', 30);
$items = $body['items'] ?? null;

if ($shopId <= 0 || !is_array($items) || count($items) === 0) {
  json_response(['ok' => false, 'error' => 'VALIDATION', 'message' => 'shop_id and items are required'], 422);
}

// Ensure shop is active + verified
$shopStmt = db()->prepare("SELECT id, name, whatsapp_phone FROM shops WHERE id=? AND is_verified=1 AND is_suspended=0 LIMIT 1");
$shopStmt->execute([$shopId]);
$shop = $shopStmt->fetch();
if (!$shop) json_response(['ok' => false, 'error' => 'NOT_FOUND', 'message' => 'Shop not found'], 404);

// Generate order code
$orderCode = bin2hex(random_bytes(6)); // 12 chars

db()->beginTransaction();

try {
  $stmt = db()->prepare("
    INSERT INTO order_intents (order_code, shop_id, customer_phone, referral_phone, status, created_at)
    VALUES (?, ?, ?, ?, 'initiated', ?)
  ");
  $stmt->execute([
    $orderCode,
    $shopId,
    $customerPhone !== '' ? $customerPhone : null,
    $referralPhone !== '' ? $referralPhone : null,
    now_utc()
  ]);

  $orderId = (int)db()->lastInsertId();

  $pStmt = db()->prepare("SELECT id, name, price FROM products WHERE id=? AND shop_id=? AND is_active=1 LIMIT 1");
  $iStmt = db()->prepare("
    INSERT INTO order_items (order_intent_id, product_id, product_name, unit_price, qty, created_at)
    VALUES (?, ?, ?, ?, ?, ?)
  ");

  foreach ($items as $it) {
    $pid = (int)($it['product_id'] ?? 0);
    $qty = (int)($it['qty'] ?? 0);

    if ($pid <= 0 || $qty <= 0) {
      throw new Exception("Invalid item");
    }

    $pStmt->execute([$pid, $shopId]);
    $p = $pStmt->fetch();
    if (!$p) {
      throw new Exception("Product not found in shop");
    }

    $iStmt->execute([$orderId, $pid, (string)$p['name'], (float)$p['price'], $qty, now_utc()]);
  }

  db()->commit();

  json_response([
    'ok' => true,
    'order_intent_id' => $orderId,
    'order_code' => $orderCode,
    'shop' => [
      'id' => (int)$shop['id'],
      'whatsapp_phone' => (string)$shop['whatsapp_phone'],
      'name' => (string)$shop['name']
    ]
  ]);
} catch (Throwable $e) {
  db()->rollBack();
  json_response(['ok' => false, 'error' => 'ORDER_CREATE_FAILED', 'message' => $e->getMessage()], 400);
}
