<?php
declare(strict_types=1);

function json_response(array $data, int $status = 200): void {
  http_response_code($status);
  echo json_encode($data, JSON_UNESCAPED_SLASHES);
  exit;
}

function get_request_json(): array {
  $raw = file_get_contents('php://input');
  if (!$raw) return [];
  $decoded = json_decode($raw, true);
  return is_array($decoded) ? $decoded : [];
}

function safe_string(mixed $v, int $maxLen = 255): string {
  $s = is_string($v) ? trim($v) : '';
  if (mb_strlen($s) > $maxLen) $s = mb_substr($s, 0, $maxLen);
  return $s;
}

function now_utc(): string {
  return gmdate('Y-m-d H:i:s');
}

function save_uploaded_image(string $fieldName, string $destDirAbs, string $destUrlPrefix): ?string {
  if (!isset($_FILES[$fieldName]) || !is_array($_FILES[$fieldName])) return null;
  $f = $_FILES[$fieldName];

  if (($f['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_NO_FILE) return null;
  if (($f['error'] ?? UPLOAD_ERR_OK) !== UPLOAD_ERR_OK) {
    json_response(['ok' => false, 'error' => 'UPLOAD_FAILED', 'message' => 'Upload error'], 400);
  }

  $tmp = $f['tmp_name'] ?? '';
  $orig = $f['name'] ?? 'file';
  $size = (int)($f['size'] ?? 0);

  if ($size > 5 * 1024 * 1024) {
    json_response(['ok' => false, 'error' => 'UPLOAD_TOO_LARGE', 'message' => 'Max 5MB'], 413);
  }

  $ext = strtolower(pathinfo((string)$orig, PATHINFO_EXTENSION));
  $allowed = ['jpg','jpeg','png','webp'];
  if (!in_array($ext, $allowed, true)) {
    json_response(['ok' => false, 'error' => 'INVALID_FILE', 'message' => 'Only jpg, png, webp allowed'], 415);
  }

  if (!is_dir($destDirAbs)) {
    @mkdir($destDirAbs, 0777, true);
  }

  $filename = bin2hex(random_bytes(16)) . '.' . $ext;
  $destAbs = rtrim($destDirAbs, '/') . '/' . $filename;

  if (!move_uploaded_file($tmp, $destAbs)) {
    json_response(['ok' => false, 'error' => 'UPLOAD_FAILED', 'message' => 'Could not move file'], 500);
  }

  return rtrim($destUrlPrefix, '/') . '/' . $filename;
}

