import { API_BASE } from "./config.js";
import { authHeaders, logout, requireRole } from "./auth.js";

const adminName = document.getElementById("admin-name");
const adminAvatar = document.getElementById("admin-avatar");
const logoutBtn = document.getElementById("logout-btn");
const messageBox = document.getElementById("admin-message");
const tabs = document.getElementById("admin-tabs");
const ownerForm = document.getElementById("create-owner-form");
const ownerResult = document.getElementById("create-owner-result");
const shopForm = document.getElementById("create-shop-form");
const shopResult = document.getElementById("create-shop-result");
const shopsList = document.getElementById("shops-list");
const shopsUnverifiedList = document.getElementById("shops-unverified");
const shopsSuspendedList = document.getElementById("shops-suspended");
const reportsList = document.getElementById("reports-list");
const ownerSelect = document.getElementById("shop-owner");
const ownerManual = document.getElementById("shop-owner-manual");
const ownersInactiveList = document.getElementById("owners-inactive");
const resetOwnerForm = document.getElementById("reset-owner-form");
const resetOwnerSelect = document.getElementById("reset-owner-select");
const resetOwnerResult = document.getElementById("reset-owner-result");
const ownerDetailsForm = document.getElementById("owner-details-form");
const ownerDetailsSelect = document.getElementById("owner-details-select");
const ownerDetailsView = document.getElementById("owner-details-view");
const shopDetailsForm = document.getElementById("shop-details-form");
const shopDetailsSelect = document.getElementById("shop-details-select");
const shopDetailsView = document.getElementById("shop-details-view");
const statProducts = document.getElementById("stat-products");
const statShops = document.getElementById("stat-shops");
const statOrders = document.getElementById("stat-orders");
const statUpdated = document.getElementById("stat-updated");

const user = requireRole("admin");
if (!user) {
  throw new Error("Unauthorized");
}
adminName.textContent = user.full_name || user.email || "Admin";
if (adminAvatar) {
  const label = adminName.textContent || "A";
  adminAvatar.textContent = label.trim().slice(0, 1).toUpperCase();
}

logoutBtn?.addEventListener("click", async () => {
  await logout();
  window.location.href = "/shop-system/app/login.html";
});

tabs?.addEventListener("click", (event) => {
  const btn = event.target.closest(".tab");
  if (!btn) return;
  const tab = btn.dataset.tab;
  document.querySelectorAll(".tab").forEach((el) => el.classList.toggle("active", el === btn));
  document.querySelectorAll(".panel").forEach((panel) => {
    panel.classList.toggle("active", panel.dataset.panel === tab);
  });
});

function showMessage(text, isError = false) {
  if (!messageBox) return;
  messageBox.textContent = text;
  messageBox.style.display = "block";
  messageBox.style.background = isError ? "#fff1f0" : "#f0fff6";
  messageBox.style.borderColor = isError ? "rgba(159, 77, 47, 0.3)" : "rgba(47, 93, 80, 0.3)";
}

async function apiRequest(path, options = {}) {
  const res = await fetch(`${API_BASE}${path}`, {
    ...options,
    headers: {
      "Content-Type": "application/json",
      ...authHeaders(),
      ...(options.headers || {})
    }
  });
  const text = await res.text();
  try {
    return JSON.parse(text);
  } catch {
    return { ok: false, error: "BAD_JSON", raw: text };
  }
}

async function apiGet(path) {
  return apiRequest(path, { method: "GET" });
}

async function apiPost(path, payload) {
  return apiRequest(path, { method: "POST", body: JSON.stringify(payload) });
}

async function loadOwners() {
  const data = await apiGet("/admin/list_owners");
  if (!data.ok) {
    ownerSelect.innerHTML = "<option value=\"\">Unable to load owners</option>";
    return;
  }
  const owners = data.owners || [];
  ownerSelect.innerHTML = owners
    .map((owner) => `<option value="${owner.id}">${owner.full_name} (#${owner.id})</option>`)
    .join("");
}

function ownerCard(owner) {
  const active = Number(owner.is_active) === 1;
  return `
    <div class="card">
      <div class="pill">#${owner.id} ${owner.full_name || "Owner"}</div>
      <p class="muted">${owner.email || owner.phone || "-"}</p>
      <div class="badge ${active ? "good" : "warn"}">
        ${active ? "Active" : "Inactive"}
      </div>
      ${active ? "" : `
        <div style="margin-top:12px; display:flex; gap:8px; flex-wrap:wrap;">
          <button class="btn secondary" data-action="activate-owner" data-id="${owner.id}">
            Reactivate
          </button>
        </div>
      `}
    </div>
  `;
}

async function loadInactiveOwners() {
  if (!ownersInactiveList) return;
  const data = await apiGet("/admin/list_owners_all");
  if (!data.ok) {
    ownersInactiveList.innerHTML = "<div class=\"notice\">Unable to load owners.</div>";
    return;
  }
  const owners = (data.owners || []).filter((owner) => Number(owner.is_active) !== 1);
  ownersInactiveList.innerHTML = owners.length
    ? owners.map(ownerCard).join("")
    : "<div class=\"notice\">No inactive owners.</div>";
}

async function loadResetOwners() {
  if (!resetOwnerSelect) return;
  const data = await apiGet("/admin/list_owners_all");
  if (!data.ok) {
    resetOwnerSelect.innerHTML = "<option value=\"\">Unable to load owners</option>";
    return;
  }
  const owners = data.owners || [];
  resetOwnerSelect.innerHTML = owners.length
    ? owners
        .map((owner) => `<option value="${owner.id}">${owner.full_name} (#${owner.id})</option>`)
        .join("")
    : "<option value=\"\">No owners found</option>";
}

async function loadOwnerDetailsOptions() {
  if (!ownerDetailsSelect) return;
  const data = await apiGet("/admin/list_owners_all");
  if (!data.ok) {
    ownerDetailsSelect.innerHTML = "<option value=\"\">Unable to load owners</option>";
    return;
  }
  const owners = data.owners || [];
  ownerDetailsSelect.innerHTML = owners.length
    ? owners
        .map((owner) => `<option value="${owner.id}">${owner.full_name} (#${owner.id})</option>`)
        .join("")
    : "<option value=\"\">No owners found</option>";
}

async function loadShopDetailsOptions() {
  if (!shopDetailsSelect) return;
  const data = await apiGet("/admin/list_shops");
  if (!data.ok) {
    shopDetailsSelect.innerHTML = "<option value=\"\">Unable to load shops</option>";
    return;
  }
  const shops = data.shops || [];
  shopDetailsSelect.innerHTML = shops.length
    ? shops
        .map((shop) => `<option value="${shop.id}">${shop.name} (#${shop.id})</option>`)
        .join("")
    : "<option value=\"\">No shops found</option>";
}

ownerSelect?.addEventListener("change", () => {
  if (ownerSelect.value) {
    ownerManual.value = ownerSelect.value;
  }
});

ownerForm?.addEventListener("submit", async (event) => {
  event.preventDefault();
  ownerResult.style.display = "none";

  const formData = new FormData(ownerForm);
  const payload = {
    full_name: String(formData.get("full_name") || "").trim(),
    phone: String(formData.get("phone") || "").trim(),
    email: String(formData.get("email") || "").trim(),
    temp_password: String(formData.get("temp_password") || "").trim()
  };

  const data = await apiPost("/admin/create_shop_owner", payload);
  if (!data.ok) {
    showMessage(data.message || data.error || "Failed to create owner.", true);
    return;
  }

  ownerResult.textContent = `Owner created. Temp password: ${data.temp_password}`;
  ownerResult.style.display = "block";
  ownerForm.reset();
  await loadOwners();
  await loadResetOwners();
});

shopForm?.addEventListener("submit", async (event) => {
  event.preventDefault();
  shopResult.style.display = "none";

  const formData = new FormData(shopForm);
  const manualOwner = String(formData.get("owner_id") || "").trim();
  const selectedOwner = ownerSelect?.value || "";
  const ownerId = manualOwner || selectedOwner;

  const payload = {
    owner_id: ownerId ? Number(ownerId) : 0,
    name: String(formData.get("name") || "").trim(),
    whatsapp_phone: String(formData.get("whatsapp_phone") || "").trim(),
    description: String(formData.get("description") || "").trim(),
    address: String(formData.get("address") || "").trim()
  };

  const data = await apiPost("/admin/create_shop", payload);
  if (!data.ok) {
    showMessage(data.message || data.error || "Failed to create shop.", true);
    return;
  }

  shopResult.textContent = `Shop created: ${data.shop?.name || payload.name}`;
  shopResult.style.display = "block";
  shopForm.reset();
  await loadShops();
});

function shopCard(shop) {
  const verified = Number(shop.is_verified) === 1;
  const suspended = Number(shop.is_suspended) === 1;
  return `
    <div class="card">
      <div class="pill">#${shop.id} ${shop.name}</div>
      <p class="muted">${shop.description || "No description"}</p>
      <div class="badge ${verified ? "good" : "warn"}">
        ${verified ? "Verified" : "Unverified"}
      </div>
      <div class="badge ${suspended ? "warn" : "good"}">
        ${suspended ? "Suspended" : "Active"}
      </div>
      <div style="margin-top:12px; display:flex; gap:8px; flex-wrap:wrap;">
        <button class="btn secondary" data-action="verify" data-id="${shop.id}" data-state="${verified ? 0 : 1}">
          ${verified ? "Unverify" : "Verify"}
        </button>
        <button class="btn secondary" data-action="suspend" data-id="${shop.id}" data-state="${suspended ? 0 : 1}">
          ${suspended ? "Unsuspend" : "Suspend"}
        </button>
      </div>
    </div>
  `;
}

function shopAttentionCard(shop) {
  const verified = Number(shop.is_verified) === 1;
  const suspended = Number(shop.is_suspended) === 1;
  const status = suspended ? "Suspended" : "Unverified";
  return `
    <div class="card">
      <div class="pill">#${shop.id} ${shop.name}</div>
      <p class="muted">${shop.description || "No description"}</p>
      <div class="badge ${suspended ? "warn" : "warn"}">${status}</div>
      <div style="margin-top:12px; display:flex; gap:8px; flex-wrap:wrap;">
        ${verified ? "" : `
          <button class="btn secondary" data-action="verify" data-id="${shop.id}" data-state="1">
            Verify
          </button>
        `}
        ${suspended ? `
          <button class="btn secondary" data-action="suspend" data-id="${shop.id}" data-state="0">
            Reactivate
          </button>
        ` : ""}
      </div>
    </div>
  `;
}

async function loadShops() {
  const data = await apiGet("/admin/list_shops");
  if (!data.ok) {
    shopsList.innerHTML = "<div class=\"notice\">Unable to load shops.</div>";
    return;
  }
  const shops = data.shops || [];
  shopsList.innerHTML = shops.map(shopCard).join("");

  if (shopsUnverifiedList) {
    const unverified = shops.filter((shop) => Number(shop.is_verified) !== 1);
    shopsUnverifiedList.innerHTML = unverified.length
      ? unverified.map(shopAttentionCard).join("")
      : "<div class=\"notice\">No unverified shops.</div>";
  }

  if (shopsSuspendedList) {
    const suspended = shops.filter((shop) => Number(shop.is_suspended) === 1);
    shopsSuspendedList.innerHTML = suspended.length
      ? suspended.map(shopAttentionCard).join("")
      : "<div class=\"notice\">No suspended shops.</div>";
  }
}

async function handleShopAction(event) {
  const btn = event.target.closest("button[data-action]");
  if (!btn) return;
  const id = Number(btn.dataset.id || 0);
  const nextState = Number(btn.dataset.state || 0);
  const action = btn.dataset.action;
  if (!id) return;

  const endpoint = action === "verify" ? "/admin/verify_shop" : "/admin/suspend_shop";
  const payload = action === "verify"
    ? { shop_id: id, is_verified: nextState }
    : { shop_id: id, is_suspended: nextState };
  const data = await apiPost(endpoint, payload);
  if (!data.ok) {
    showMessage(data.message || data.error || "Failed to update shop.", true);
    return;
  }
  await loadShops();
}

shopsList?.addEventListener("click", handleShopAction);
shopsUnverifiedList?.addEventListener("click", handleShopAction);
shopsSuspendedList?.addEventListener("click", handleShopAction);

ownersInactiveList?.addEventListener("click", async (event) => {
  const btn = event.target.closest("button[data-action=\"activate-owner\"]");
  if (!btn) return;
  const id = Number(btn.dataset.id || 0);
  if (!id) return;

  const data = await apiPost("/admin/activate_owner", { owner_id: id });
  if (!data.ok) {
    showMessage(data.message || data.error || "Failed to reactivate owner.", true);
    return;
  }
  await loadInactiveOwners();
  await loadOwners();
  await loadResetOwners();
});

resetOwnerForm?.addEventListener("submit", async (event) => {
  event.preventDefault();
  if (!resetOwnerSelect || !resetOwnerResult) return;
  resetOwnerResult.style.display = "none";

  const ownerId = Number(resetOwnerSelect.value || 0);
  if (!ownerId) {
    resetOwnerResult.textContent = "Select an owner first.";
    resetOwnerResult.style.display = "block";
    return;
  }

  const data = await apiPost("/admin/reset_owner_password", { owner_id: ownerId });
  if (!data.ok) {
    showMessage(data.message || data.error || "Failed to reset password.", true);
    return;
  }

  resetOwnerResult.textContent = `Temp password: ${data.temp_password}`;
  resetOwnerResult.style.display = "block";
});

ownerDetailsForm?.addEventListener("submit", async (event) => {
  event.preventDefault();
  if (!ownerDetailsSelect || !ownerDetailsView) return;
  ownerDetailsView.style.display = "none";

  const ownerId = Number(ownerDetailsSelect.value || 0);
  if (!ownerId) {
    ownerDetailsView.textContent = "Select an owner first.";
    ownerDetailsView.style.display = "block";
    return;
  }

  const data = await apiGet(`/admin/owner_details?owner_id=${encodeURIComponent(ownerId)}`);
  if (!data.ok) {
    showMessage(data.message || data.error || "Failed to load owner details.", true);
    return;
  }

  const owner = data.owner || {};
  const shops = data.shops || [];
  ownerDetailsView.innerHTML = `
    <strong>${owner.full_name || "Owner"}</strong><br />
    Email: ${owner.email || "-"}<br />
    Phone: ${owner.phone || "-"}<br />
    Status: ${Number(owner.is_active) === 1 ? "Active" : "Inactive"}<br />
    Must change password: ${Number(owner.must_change_password) === 1 ? "Yes" : "No"}<br />
    Created: ${owner.created_at || "-"}<br />
    Shops: ${shops.length ? shops.map((s) => `${s.name} (#${s.id})`).join(", ") : "None"}
  `;
  ownerDetailsView.style.display = "block";
});

shopDetailsForm?.addEventListener("submit", async (event) => {
  event.preventDefault();
  if (!shopDetailsSelect || !shopDetailsView) return;
  shopDetailsView.style.display = "none";

  const shopId = Number(shopDetailsSelect.value || 0);
  if (!shopId) {
    shopDetailsView.textContent = "Select a shop first.";
    shopDetailsView.style.display = "block";
    return;
  }

  const data = await apiGet(`/admin/shop_details?shop_id=${encodeURIComponent(shopId)}`);
  if (!data.ok) {
    showMessage(data.message || data.error || "Failed to load shop details.", true);
    return;
  }

  const shop = data.shop || {};
  const owner = data.owner || {};
  shopDetailsView.innerHTML = `
    <strong>${shop.name || "Shop"}</strong><br />
    Owner: ${owner.full_name || "-"} (#${owner.id || "-"})<br />
    WhatsApp: ${shop.whatsapp_phone || "-"}<br />
    Address: ${shop.address || "-"}<br />
    Verified: ${Number(shop.is_verified) === 1 ? "Yes" : "No"}<br />
    Suspended: ${Number(shop.is_suspended) === 1 ? "Yes" : "No"}<br />
    Created: ${shop.created_at || "-"}
  `;
  shopDetailsView.style.display = "block";
});

async function loadReports() {
  const data = await apiGet("/admin/reports_list");
  if (!data.ok) {
    reportsList.innerHTML = "<div class=\"notice\">Unable to load reports.</div>";
    return;
  }
  const reports = data.reports || [];
  if (!reports.length) {
    reportsList.innerHTML = "<div class=\"notice\">No open reports.</div>";
    return;
  }
  reportsList.innerHTML = `
    <table class="table">
      <thead>
        <tr>
          <th>ID</th>
          <th>Type</th>
          <th>Reason</th>
          <th>Message</th>
          <th>Reporter</th>
          <th>Status</th>
        </tr>
      </thead>
      <tbody>
        ${reports.map((report) => `
          <tr>
            <td>#${report.id}</td>
            <td>${report.target_type} ${report.target_id}</td>
            <td>${report.reason}</td>
            <td>${report.message || "-"}</td>
            <td>${report.reporter_phone || "-"}</td>
            <td>${report.status}</td>
          </tr>
        `).join("")}
      </tbody>
    </table>
  `;
}

function setStat(el, value, fallback = "--") {
  if (!el) return;
  el.textContent = value === null || value === undefined ? fallback : String(value);
}

async function loadStats() {
  const data = await apiGet("/admin/dashboard_stats");
  if (!data.ok) {
    setStat(statProducts, null);
    setStat(statShops, null);
    setStat(statOrders, null);
    if (statUpdated) statUpdated.textContent = "Unavailable";
    return;
  }

  const stats = data.stats || {};
  setStat(statProducts, stats.total_products ?? 0);
  setStat(statShops, stats.total_shops ?? 0);
  setStat(statOrders, stats.completed_orders ?? 0);
  if (statUpdated) statUpdated.textContent = new Date().toLocaleString();
}

loadOwners();
loadShops();
loadReports();
loadStats();
loadInactiveOwners();
loadResetOwners();
loadOwnerDetailsOptions();
loadShopDetailsOptions();
