import { API_BASE } from "./config.js";
import { apiGet } from "./api.js";
import { addToCart, getCart, cartCount } from "./cart.js";



// Helpers
const qs = (sel) => document.querySelector(sel);
const money = (v) => {
  const n = Number(v || 0);
  return new Intl.NumberFormat("en-NG", { style: "currency", currency: "NGN", maximumFractionDigits: 0 }).format(n);
};

const pickTopShops = (shops, limit) => {
  const hasReviews = shops.some((shop) => Number(shop.review_count || 0) > 0);
  if (!hasReviews) return shops.slice(0, limit);

  return [...shops]
    .sort((a, b) => {
      const ratingA = Number(a.avg_rating || 0);
      const ratingB = Number(b.avg_rating || 0);
      if (ratingB !== ratingA) return ratingB - ratingA;
      const countA = Number(a.review_count || 0);
      const countB = Number(b.review_count || 0);
      if (countB !== countA) return countB - countA;
      return Number(b.id || 0) - Number(a.id || 0);
    })
    .slice(0, limit);
};

function shopCard(shop) {
  return `
    <div class="card">
      <div class="card-head">
        <div class="avatar">${(shop.name || "S").slice(0,1).toUpperCase()}</div>
        <div class="meta">
          <div class="title">${shop.name}</div>
          <div class="sub">${shop.address || ""}</div>
        </div>
      </div>
      <div class="card-body">
        <p class="desc">${shop.description || ""}</p>
        <a class="btn" href="shop.html?shop_id=${shop.id}">View Shop</a>
      </div>
    </div>
  `;
}

function productCard(p) {
  const img = p.image_path ? `<img class="pimg" src="${p.image_path}" alt="">` : `<div class="pimg ph"></div>`;
  return `
    <div class="card">
      ${img}
      <div class="card-body">
        <div class="title">${p.name}</div>
        <div class="sub">${p.shop_name ? p.shop_name : ""}</div>
        <div class="price">${money(p.price)}</div>
        <a class="btn" href="product.html?product_id=${p.id}">View Product</a>
      </div>
    </div>
  `;
}

// INDEX PAGE
export async function initIndex() {
  const shopsWrap = qs("#shopsGrid");
  const productsWrap = qs("#productsGrid");
  let allShops = [];
  let allProducts = [];

  shopsWrap.innerHTML = `<div class="muted">Loading shops...</div>`;
  productsWrap.innerHTML = `<div class="muted">Loading products...</div>`;

  const [shopsRes, prodRes] = await Promise.all([
    apiGet(`${API_BASE}/public/list_shops`),
    apiGet(`${API_BASE}/public/list_products?limit=60`),
  ]);

  if (!shopsRes.ok) {
    shopsWrap.innerHTML = `<div class="muted">Failed to load shops.</div>`;
  } else {
    allShops = shopsRes.shops || [];
    const topShops = pickTopShops(allShops, 4);
    shopsWrap.innerHTML = topShops.map(shopCard).join("") || `<div class="muted">No shops found.</div>`;
  }

  if (!prodRes.ok) {
    productsWrap.innerHTML = `<div class="muted">Failed to load products.</div>`;
  } else {
    allProducts = prodRes.products || [];
    productsWrap.innerHTML = allProducts.map(productCard).join("") || `<div class="muted">No products found.</div>`;
  }

  // Search products
  const searchForm = qs("#searchForm");
  const searchInput = qs("#searchInput");

  if (searchForm && searchInput) {
    searchForm.addEventListener("submit", async (e) => {
      e.preventDefault();
      const q = searchInput.value.trim();
      const term = q.toLowerCase();

      if (!term) {
        const topShops = pickTopShops(allShops, 4);
        shopsWrap.innerHTML = topShops.map(shopCard).join("") || `<div class="muted">No shops found.</div>`;
        productsWrap.innerHTML = allProducts.map(productCard).join("") || `<div class="muted">No products found.</div>`;
        return;
      }

      const productMatches = allProducts.filter((p) =>
        [p.name, p.description, p.shop_name].some((v) => (v || "").toLowerCase().includes(term))
      );
      const shopMatches = allShops.filter((s) =>
        [s.name, s.address, s.description].some((v) => (v || "").toLowerCase().includes(term))
      );

      const topMatches = pickTopShops(shopMatches, 4);
      productsWrap.innerHTML = productMatches.map(productCard).join("") || `<div class="muted">No products for "${q}".</div>`;
      shopsWrap.innerHTML = topMatches.map(shopCard).join("") || `<div class="muted">No shops for "${q}".</div>`;
    });
  }
}

// SHOP PAGE
export async function initShop() {
  const params = new URLSearchParams(location.search);
  const shopId = params.get("shop_id");

  const header = qs("#shopHeader");
  const grid = qs("#shopProducts");

  header.innerHTML = `<div class="muted">Loading shop...</div>`;
  grid.innerHTML = `<div class="muted">Loading products...</div>`;

  const res = await apiGet(`${API_BASE}/public/shop_details?shop_id=${encodeURIComponent(shopId || "")}`);
  if (!res.ok) {
    header.innerHTML = `<div class="muted">Shop not found.</div>`;
    grid.innerHTML = "";
    return;
  }

  const s = res.shop;
  header.innerHTML = `
    <div class="shop-hero">
      <div>
        <h1>${s.name}</h1>
        <div class="muted">${s.address || ""}</div>
        <p>${s.description || ""}</p>
      </div>
      <div class="actions">
        <a class="btn" href="https://wa.me/${s.whatsapp_phone}" target="_blank" rel="noreferrer">Chat on WhatsApp</a>
        <a class="btn ghost" href="index.html">Back</a>
      </div>
    </div>
  `;

  const products = res.products || [];
  grid.innerHTML = products.map(p => productCard({ ...p, shop_name: s.name })).join("") || `<div class="muted">No products yet.</div>`;
}

// PRODUCT PAGE
export async function initProduct() {
  const wrap = qs("#productWrap");
  if (!wrap) return;

  const params = new URLSearchParams(location.search);
  const productId = params.get("product_id");

  if (!productId) {
    wrap.innerHTML = `<div class="muted">Product not found.</div>`;
    return;
  }

  wrap.innerHTML = `<div class="muted">Loading product...</div>`;

  let p;
  try {
    const res = await apiGet(`${API_BASE}/public/product_details?product_id=${encodeURIComponent(productId)}`);
    if (!res.ok) {
      wrap.innerHTML = `<div class="muted">Product not found.</div>`;
      return;
    }
    p = res.product;
  } catch {
    wrap.innerHTML = `<div class="muted">Failed to load product.</div>`;
    return;
  }

  const img = p.image_path
    ? `<img class="pimg-lg" src="${p.image_path}" alt="">`
    : `<div class="pimg-lg ph"></div>`;

  const count = cartCount(getCart());

  wrap.innerHTML = `
    <div class="product-page">
      <div class="media">${img}</div>
      <div class="info">
        <h1>${p.name}</h1>
        <div class="price">${money(p.price)}</div>
        <div class="muted">Shop: ${p.shop_name}</div>
        <p>${p.description || ""}</p>

        <div class="actions">
          <button id="addToCartBtn" class="btn">Add to Cart</button>
          <a class="btn ghost" href="cart.html">Cart (${count})</a>
          <a class="btn ghost" href="shop.html?shop_id=${p.shop_id}">Back to Shop</a>
        </div>
      </div>
    </div>
  `;

  const addBtn = qs("#addToCartBtn");
  if (!addBtn) return;
  addBtn.addEventListener("click", () => {
    const shop = { id: p.shop_id, name: p.shop_name, whatsapp_phone: p.whatsapp_phone };
    const product = { id: p.id, name: p.name, price: p.price, image_path: p.image_path };
    const r = addToCart(shop, product, 1);
    if (!r.ok && r.error === "ONE_SHOP_ONLY") alert(r.message);
    else alert("Added to cart!");
  });
}

// === Premium UI helpers ===
export async function initCartBadge() {
  try {
    const mod = await import("./cart.js");
    const badge = document.querySelector("#cartBadge");
    if (!badge) return;

    const cart = mod.getCart();
    const count = cart ? mod.cartCount(cart) : 0;

    if (count > 0) {
      badge.style.display = "flex";
      badge.textContent = String(count);
    } else {
      badge.style.display = "none";
    }
  } catch {}
}
