CREATE DATABASE IF NOT EXISTS daura_shop
  CHARACTER SET utf8mb4
  COLLATE utf8mb4_unicode_ci;

USE daura_shop;

-- USERS (admin + owners only)
CREATE TABLE IF NOT EXISTS users (
  id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
  role ENUM('admin','owner') NOT NULL,
  full_name VARCHAR(150) NOT NULL,
  phone VARCHAR(30) DEFAULT NULL,
  email VARCHAR(150) DEFAULT NULL,
  password_hash VARCHAR(255) NOT NULL,
  must_change_password TINYINT(1) NOT NULL DEFAULT 0,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NULL,
  UNIQUE KEY uniq_email (email),
  KEY idx_role (role)
);

-- SHOPS
CREATE TABLE IF NOT EXISTS shops (
  id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
  owner_id BIGINT UNSIGNED NOT NULL,
  name VARCHAR(180) NOT NULL,
  description TEXT NULL,
  whatsapp_phone VARCHAR(30) NOT NULL,
  logo_path VARCHAR(255) NULL,
  address VARCHAR(255) NULL,
  is_verified TINYINT(1) NOT NULL DEFAULT 0,
  is_suspended TINYINT(1) NOT NULL DEFAULT 0,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NULL,
  FOREIGN KEY (owner_id) REFERENCES users(id)
    ON DELETE RESTRICT ON UPDATE CASCADE,
  KEY idx_owner (owner_id),
  KEY idx_verified (is_verified),
  KEY idx_suspended (is_suspended)
);

-- PRODUCTS
CREATE TABLE IF NOT EXISTS products (
  id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
  shop_id BIGINT UNSIGNED NOT NULL,
  name VARCHAR(200) NOT NULL,
  description TEXT NULL,
  price DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  image_path VARCHAR(255) NULL,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NULL,
  FOREIGN KEY (shop_id) REFERENCES shops(id)
    ON DELETE RESTRICT ON UPDATE CASCADE,
  KEY idx_shop (shop_id),
  KEY idx_active (is_active)
);

-- ORDER INTENTS (guest checkout)
CREATE TABLE IF NOT EXISTS order_intents (
  id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
  order_code VARCHAR(30) NOT NULL,
  shop_id BIGINT UNSIGNED NOT NULL,
  status ENUM('initiated','completed') NOT NULL DEFAULT 'initiated',
  customer_phone VARCHAR(30) NULL,
  referral_phone VARCHAR(30) NULL,
  total_estimate DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  created_at DATETIME NOT NULL,
  completed_at DATETIME NULL,
  FOREIGN KEY (shop_id) REFERENCES shops(id)
    ON DELETE RESTRICT ON UPDATE CASCADE,
  UNIQUE KEY uniq_order_code (order_code),
  KEY idx_shop_status (shop_id, status)
);

-- ORDER ITEMS snapshot
CREATE TABLE IF NOT EXISTS order_items (
  id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
  order_intent_id BIGINT UNSIGNED NOT NULL,
  product_id BIGINT UNSIGNED NULL,
  product_name VARCHAR(200) NOT NULL,
  unit_price DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  qty INT UNSIGNED NOT NULL DEFAULT 1,
  created_at DATETIME NOT NULL,
  FOREIGN KEY (order_intent_id) REFERENCES order_intents(id)
    ON DELETE CASCADE ON UPDATE CASCADE,
  KEY idx_order (order_intent_id)
);

-- REVIEW TOKENS (one-time)
CREATE TABLE IF NOT EXISTS review_tokens (
  id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
  token VARCHAR(80) NOT NULL,
  order_intent_id BIGINT UNSIGNED NOT NULL,
  shop_id BIGINT UNSIGNED NOT NULL,
  is_used TINYINT(1) NOT NULL DEFAULT 0,
  expires_at DATETIME NOT NULL,
  created_at DATETIME NOT NULL,
  used_at DATETIME NULL,
  FOREIGN KEY (order_intent_id) REFERENCES order_intents(id)
    ON DELETE RESTRICT ON UPDATE CASCADE,
  FOREIGN KEY (shop_id) REFERENCES shops(id)
    ON DELETE RESTRICT ON UPDATE CASCADE,
  UNIQUE KEY uniq_token (token),
  KEY idx_shop_used (shop_id, is_used)
);

-- REVIEWS (verified only)
CREATE TABLE IF NOT EXISTS reviews (
  id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
  shop_id BIGINT UNSIGNED NOT NULL,
  order_intent_id BIGINT UNSIGNED NOT NULL,
  rating TINYINT UNSIGNED NOT NULL,
  comment TEXT NULL,
  created_at DATETIME NOT NULL,
  FOREIGN KEY (shop_id) REFERENCES shops(id)
    ON DELETE RESTRICT ON UPDATE CASCADE,
  FOREIGN KEY (order_intent_id) REFERENCES order_intents(id)
    ON DELETE RESTRICT ON UPDATE CASCADE,
  KEY idx_shop (shop_id),
  KEY idx_rating (rating)
);

-- REPORTS (guest)
CREATE TABLE IF NOT EXISTS reports (
  id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
  target_type ENUM('shop','product') NOT NULL,
  target_id BIGINT UNSIGNED NOT NULL,
  reason VARCHAR(120) NOT NULL,
  message TEXT NULL,
  reporter_phone VARCHAR(30) NULL,
  status ENUM('open','resolved') NOT NULL DEFAULT 'open',
  created_at DATETIME NOT NULL,
  resolved_at DATETIME NULL,
  KEY idx_target (target_type, target_id),
  KEY idx_status (status)
);
